import { Request, Response } from 'express';
import { marketPricesService } from './service';

function hasFeature(user: any, featureKey: string): boolean {
  return user?.features?.includes(featureKey);
}

export const marketPricesController = {
  // POST /market-prices
  async submit(req: any, res: Response) {
    const user = req.user;
    if (!user) return res.status(401).json({ error: 'Unauthorized' });
    if (!user.can_update_prices && user.role !== 'admin') {
      return res.status(403).json({ error: 'Not authorized to submit prices.' });
    }
    const { data, error } = await marketPricesService.submitMarketPrice(user, req.body);
    if (error) return res.status(400).json({ error: error.message });
    return res.status(201).json({ success: true, data });
  },

  // GET /market-prices
  async list(req: any, res: Response) {
    const user = req.user;
    if (!user) return res.status(401).json({ error: 'Unauthorized' });
    const { market_id, product_id, submitted_by, date_range } = req.query;
    const filters: any = { market_id, product_id, submitted_by, date_range };
    const { data, error } = await marketPricesService.listMarketPrices(filters);
    if (error) return res.status(400).json({ error: error.message });
    return res.json({ success: true, data });
  },

  // GET /market-prices/suggestions
  async suggestions(req: any, res: Response) {
    const user = req.user;
    if (!user) return res.status(401).json({ error: 'Unauthorized' });
    if (!hasFeature(user, 'market_price_suggestions')) {
      return res.status(403).json({ error: 'Market price suggestions not available on your plan.' });
    }
    const { data, error } = await marketPricesService.getSuggestions(user.vendor || user);
    if (error) return res.status(400).json({ error: error.message });
    return res.json({ success: true, data });
  },

  // PATCH /admin/users/:id/pricing-access
  async updatePricingAccess(req: any, res: Response) {
    const admin = req.admin;
    if (!admin || admin.role !== 'super_admin') {
      return res.status(403).json({ error: 'Only super admin can update pricing access.' });
    }
    const { id } = req.params;
    const { data, error } = await marketPricesService.updatePricingAccess(id, req.body);
    if (error) return res.status(400).json({ error: error.message });
    return res.json({ success: true, data });
  }
}; 